#include "../SimulationEngine/SimulationEngineController.h"


#ifndef __DRAWSTUFF_H__
#define __DRAWSTUFF_H__

/* Define a DLL export symbol for those platforms that need it */
#if defined(ODE_PLATFORM_WINDOWS)
  #if defined(DS_DLL)
    #define DS_API __declspec(dllexport)
  #elif !defined(DS_LIB)
    #define DS_DLL_API __declspec(dllimport)
  #endif
#endif
    
#if !defined(DS_API)
  #define DS_API
#endif

#ifdef __cplusplus
extern "C" {
#endif

/* texture numbers */
#define DS_NONE   0	/* uses the current color instead of a texture */
#define DS_WOOD   1

DS_API void dsSimulationLoop (int window_width, int window_height);

DS_API void dsError (char *msg, ...);

DS_API void dsDebug (char *msg, ...);

DS_API void dsPrint (char *msg, ...);

/**
 * @brief Sets the viewpoint
 * @ingroup drawstuff
 * @param xyz camera position.
 * @param hpr contains heading, pitch and roll numbers in degrees. heading=0
 * points along the x axis, pitch=0 is looking towards the horizon, and
 * roll 0 is "unrotated".
 */
DS_API void dsSetViewpoint (float xyz[3], float hpr[3]);


/**
 * @brief Gets the viewpoint
 * @ingroup drawstuff
 * @param xyz position
 * @param hpr heading,pitch,roll.
 */
DS_API void dsGetViewpoint (float xyz[3], float hpr[3]);

/**
 * @brief Stop the simulation loop.
 * @ingroup drawstuff
 * Calling this from within dsSimulationLoop()
 * will cause it to exit and return to the caller. it is the same as if the
 * user used the exit command. using this outside the loop will have no
 * effect.
 */
DS_API void dsStop();

/**
 * @brief Get the elapsed time (on wall-clock)
 * @ingroup drawstuff
 * It returns the nr of seconds since the last call to this function.
 */
DS_API double dsElapsedTime();

/**
 * @brief Set the color with which geometry is drawn.
 * @ingroup drawstuff
 * @param red Red component from 0 to 1
 * @param green Green component from 0 to 1
 * @param blue Blue component from 0 to 1
 */
DS_API void dsSetColor (float red, float green, float blue);

/**
 * @brief Set the color and transparency with which geometry is drawn.
 * @ingroup drawstuff
 * @param alpha Note that alpha transparency is a misnomer: it is alpha opacity.
 * 1.0 means fully opaque, and 0.0 means fully transparent.
 */
DS_API void dsSetColorAlpha (float red, float green, float blue, float alpha);

/**
 * @brief Draw a box.
 * @ingroup drawstuff
 * @param pos is the x,y,z of the center of the object.
 * @param R is a 3x3 rotation matrix for the object, stored by row like this:
 *        [ R11 R12 R13 0 ]
 *        [ R21 R22 R23 0 ]
 *        [ R31 R32 R33 0 ]
 * @param sides[] is an array of x,y,z side lengths.
 */
DS_API void dsDrawBox (const float pos[3], const float R[12], const float sides[3]);

/**
 * @brief Draw a sphere.
 * @ingroup drawstuff
 * @param pos Position of center.
 * @param R orientation.
 * @param radius
 */
DS_API void dsDrawSphere (const float pos[3], const float R[12], float radius);

/**
 * @brief Draw a triangle.
 * @ingroup drawstuff
 * @param pos Position of center
 * @param R orientation
 * @param v0 first vertex
 * @param v1 second
 * @param v2 third vertex
 * @param solid set to 0 for wireframe
 */
DS_API void dsDrawTriangle (const float pos[3], const float R[12],
		     const float *v0, const float *v1, const float *v2, int solid);

/**
 * @brief Draw a z-aligned cylinder
 * @ingroup drawstuff
 */
DS_API void dsDrawCylinder (const float pos[3], const float R[12],
		     float length, float radius);

/**
 * @brief Draw a z-aligned capsule
 * @ingroup drawstuff
 */
DS_API void dsDrawCapsule (const float pos[3], const float R[12],
		    float length, float radius);

/**
 * @brief Draw a line.
 * @ingroup drawstuff
 */
DS_API void dsDrawLine (const float pos1[3], const float pos2[3]);

/**
 * @brief Draw a convex shape.
 * @ingroup drawstuff
 */
DS_API void dsDrawConvex(const float pos[3], const float R[12],
		  float *_planes,
		  unsigned int _planecount,
		  float *_points,
		  unsigned int _pointcount,
		  unsigned int *_polygons);

 /* these drawing functions are identical to the ones above, except they take
 * double arrays for `pos' and `R'.
 */
DS_API void dsDrawBoxD (const double pos[3], const double R[12],
		 const double sides[3]);
DS_API void dsDrawSphereD (const double pos[3], const double R[12],
		    const float radius);
DS_API void dsDrawTriangleD (const double pos[3], const double R[12],
		      const double *v0, const double *v1, const double *v2, int solid);
DS_API void dsDrawCylinderD (const double pos[3], const double R[12],
		      float length, float radius);
DS_API void dsDrawCapsuleD (const double pos[3], const double R[12],
		     float length, float radius);
DS_API void dsDrawLineD (const double pos1[3], const double pos2[3]);
DS_API void dsDrawConvexD(const double pos[3], const double R[12],
		  double *_planes,
		  unsigned int _planecount,
		  double *_points,
		  unsigned int _pointcount,
		  unsigned int *_polygons);

/**
 * @brief Set the quality with which curved objects are rendered.
 * @ingroup drawstuff
 * Higher numbers are higher quality, but slower to draw. 
 * This must be set before the first objects are drawn to be effective.
 * Default sphere quality is 1, default capsule quality is 3.
 */
DS_API void dsSetSphereQuality (int n);		/* default = 1 */
DS_API void dsSetCapsuleQuality (int n);		/* default = 3 */

// Backwards compatible API
#define dsDrawCappedCylinder dsDrawCapsule
#define dsDrawCappedCylinderD dsDrawCapsuleD
#define dsSetCappedCylinderQuality dsSetCapsuleQuality

/* closing bracket for extern "C" */
#ifdef __cplusplus
}
#endif

#endif

